#!/usr/bin/perl -w
#
# Storage module for Acctfind.pm based on regular file
#
use strict;

package Port25::Acctfind::Storage::File;

use Port25::Acctfind::Storage;

use vars qw(@ISA);

@ISA = ("Port25::Acctfind::Storage");


use POSIX;


sub new {
    my $class = shift @_;
    my $tmpDir = shift@_;
    
    my $self = $class->SUPER::new($tmpDir);
    
    $self->{acctfind} = shift @_;
    
    local *DB_Handle;
    $self->{DB_HANDLE} = *DB_Handle;
    
    sysopen DB_Handle, "$self->{dbFile}", O_RDWR|O_CREAT|O_EXCL
        or die "$0: could not open DB file $self->{dbFile}: $!\n";
        
    binmode $self->{DB_HANDLE};
    
    $self->{dbEntries} = 0;
    $self->{dbEntryCounter} = "a" x 8, # About 8 billion
    
    return $self;
}


sub DESTROY {
    my $self = shift @_;
    
    close $self->{DB_HANDLE} if $self->{DB_HANDLE};
    
    $self->SUPER::DESTROY(@_);
}


# we do not support sorting
sub setupSorting {
}


# rewind: called to read DB file after writing
sub rewindDb {
    my $self = shift;
    
    # seek to beginning of DB file
    sysseek $self->{DB_HANDLE}, 0, SEEK_SET or die "$0: $!";
}


#
# Loads a record from the data store
#
# @param  $key  the key of the last item returned, or '' if first call
# @param  $n    0 for the first call, !0 for succeeding calls
# @return  $status  0 for success, !0 for errors
# @return  $key     unused
# @return  $rec     the record loaded
#
sub loadRecord {
    my $self = shift @_;
    my $key = shift @_;
    my $n = shift @_;
    
    my $chunk = '';
    my $status = 0;
    
    $self->rewindDb() if $n == 0;
    
    # sysread frozen record from data file
    my $b;                              # bytes read
    my $l;                              # length of data chunk to read
    
    # read packed record's length, then data chunk
    $l = 4;                             # size of the packed length ("L")
    
    unless (($b = sysread $self->{DB_HANDLE}, $chunk, $l) == $l) {
        $status = 1;                      # as DB_File does
        
        return $status, $key, undef if defined $b and $b == 0; # if eof
        
        warn "$l != $b";
        $status = -1;
    }
    
    $l = unpack "L", $chunk;
    unless (($b = sysread $self->{DB_HANDLE}, $chunk, $l) == $l) {
        warn "$l != $b";
        $status = -1;
    }
    
    my $rec = $self->{acctfind}->thawRec($chunk) if $chunk and $status == 0;
    
    return $status, $key, $rec;
}


sub storeRecord {
    my $self = shift @_;
    my $rec = shift @_;
    
    my $db = $self->{db};
    my $status = 0;
    my $key;
    
    $self->{dbEntryCounter}++;
    $key = $self->{dbEntryCounter};
    
    my $chunk = $self->{acctfind}->freezeRec($rec);
    
    # syswrite frozen record to DB file
    my $b;                          # bytes written
    my $l = length $chunk;
    unless (($b = syswrite $self->{DB_HANDLE}, pack( "L", $l), 4) == 4) {
        warn "4 != $b";
        $status = -1;
    }
    unless (($b = syswrite $self->{DB_HANDLE}, $chunk, $l) == $l) {
        warn "$l != $b";
        $status = -1;
    }
    
    $self->{dbEntries}++;
    
    return $status < 0
           ? "An error occured while writing to file $self->{dbFile}.\n"
           : undef;
}

1;

